/*
 * Copyright (c) 2009-2012, Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Module Name:    mfw_gst_v4l.c
 *
 * Description:    Implementation of V4L CORE functions for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */

/*
 * Changelog:
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <errno.h>
#include <gst/gst.h>

#include "mfw_gst_v4l.h"
#include "mfw_gst_v4l_buffer.h"
#include "mfw_gst_v4lsink.h"


/*=============================================================================
                             MACROS
=============================================================================*/


/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/
GST_DEBUG_CATEGORY_EXTERN (mfw_gst_v4lsink_debug);
#define GST_CAT_DEFAULT mfw_gst_v4lsink_debug


/*=============================================================================
                             GLOBAL FUNCTIONS
=============================================================================*/

extern guint mfw_gst_v4lsink_signals[SIGNAL_LAST];

/*=============================================================================
FUNCTION:           dumpfile_open

DESCRIPTION:        This function will open the location file.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
dumpfile_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  /* open the file */
  if (v4l_info->dump_location == NULL || v4l_info->dump_location[0] == '\0')
    goto no_dumpfilename;

  v4l_info->dumpfile = fopen (v4l_info->dump_location, "wb");
  if (v4l_info->dumpfile == NULL)
    goto open_failed;

  v4l_info->dump_length = 0;

  GST_DEBUG_OBJECT (v4l_info, "opened file %s", v4l_info->dump_location);

  return TRUE;

  /* ERRORS */
no_dumpfilename:
  {
    GST_ERROR ("No file name specified for dumping.");
    return FALSE;
  }
open_failed:
  {
    GST_ERROR ("Could not open file \"%s\" for writing.",
        v4l_info->dump_location);
    return FALSE;
  }
}

/*=============================================================================
FUNCTION:           dumpfile_close

DESCRIPTION:        This function will close the location file.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void
dumpfile_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  if (v4l_info->dumpfile) {
    if (fclose (v4l_info->dumpfile) != 0)
      goto close_failed;

    GST_DEBUG_OBJECT (v4l_info, "closed file");
    v4l_info->dumpfile = NULL;
  }
  return;

  /* ERRORS */
close_failed:
  {
    GST_ERROR ("Error closing file:%s", v4l_info->dump_location);
    return;
  }
}

/*=============================================================================
FUNCTION:           dumpfile_set_location

DESCRIPTION:        This function initialize the dump file environment.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
dumpfile_set_location (MFW_GST_V4LSINK_INFO_T * v4l_info,
    const gchar * location)
{
  if (v4l_info->dumpfile)
    goto was_open;

  g_free (v4l_info->dump_location);
  if (location != NULL) {
    v4l_info->enable_dump = TRUE;
    v4l_info->dump_location = g_strdup (location);
  } else {
    v4l_info->enable_dump = FALSE;
    v4l_info->dump_location = NULL;
  }

  return TRUE;

  /* ERRORS */
was_open:
  {
    g_warning ("Changing the `dump_location' property on v4lsink when "
        "a file is open not supported.");
    return FALSE;
  }
}

/*=============================================================================
FUNCTION:           dumpfile_write

DESCRIPTION:        This function write the image to file.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
dumpfile_write (MFW_GST_V4LSINK_INFO_T * v4l_info, GstBuffer * buffer)
{
  guint64 cur_pos;
  guint size;

  size = GST_BUFFER_SIZE (buffer);

  cur_pos = v4l_info->dump_length;

  GST_DEBUG_OBJECT (v4l_info, "writing %u bytes at %" G_GUINT64_FORMAT,
      size, cur_pos);

  if (size > 0 && GST_BUFFER_DATA (buffer) != NULL) {
    if (v4l_info->cr_left_bypixel != 0 || v4l_info->cr_right_bypixel != 0
        || v4l_info->cr_top_bypixel != 0 || v4l_info->cr_bottom_bypixel != 0) {
      /* remove black edge */
      gint y;
      char *p;
      gint cr_left = v4l_info->cr_left_bypixel_orig;
      gint cr_right = v4l_info->cr_right_bypixel_orig;
      gint cr_top = v4l_info->cr_top_bypixel_orig;
      gint cr_bottom = v4l_info->cr_bottom_bypixel_orig;
      gint stride = v4l_info->width + cr_left + cr_right;

      /* Y */
      for (y = cr_top; y < v4l_info->height + cr_top; y++) {
        p = (char *) (GST_BUFFER_DATA (buffer)) + y * stride + cr_left;
        fwrite (p, 1, v4l_info->width, v4l_info->dumpfile);
        v4l_info->dump_length += v4l_info->width;
      }

      /* U */
      for (y = cr_top / 2; y < (v4l_info->height + cr_top) / 2; y++) {
        p = (char *) (GST_BUFFER_DATA (buffer)) +
            stride * (v4l_info->height + cr_top + cr_bottom) +
            (y * stride + cr_left) / 2;
        fwrite (p, 1, v4l_info->width / 2, v4l_info->dumpfile);
        v4l_info->dump_length += (v4l_info->width / 2);
      }

      /* V */
      for (y = cr_top / 2; y < (v4l_info->height + cr_top) / 2; y++) {
        p = (char *) (GST_BUFFER_DATA (buffer)) +
            stride * (v4l_info->height + cr_top + cr_bottom) * 5 / 4 +
            (y * stride + cr_left) / 2;
        fwrite (p, 1, v4l_info->width / 2, v4l_info->dumpfile);
        v4l_info->dump_length += (v4l_info->width / 2);
      }
    } else {
      if (fwrite (GST_BUFFER_DATA (buffer), size, 1, v4l_info->dumpfile)
          != 1)
        goto handle_error;

      v4l_info->dump_length += size;
    }
  }

  return TRUE;

handle_error:
  {
    switch (errno) {
      case ENOSPC:{
        GST_ELEMENT_ERROR (v4l_info, RESOURCE, NO_SPACE_LEFT, (NULL), (NULL));
        break;
      }
      default:{
        GST_ELEMENT_ERROR (v4l_info, RESOURCE, WRITE,
            (("Error while writing to file \"%s\"."),
                v4l_info->dump_location), ("%s", g_strerror (errno)));
      }
    }
    return FALSE;
  }
}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_open

DESCRIPTION:        This function open the default v4l device.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gboolean retval = TRUE;

  /*No need to open v4l device when it has opened--change para on-the-fly */
    if (v4l_info->v4l_id<0){

      if (v4l_info->v4l_dev_name[0]!='\0'){
        v4l_info->v4l_id =
                  open (v4l_info->v4l_dev_name, O_RDWR | O_NONBLOCK, 0);
      }else{
        v4l_info->v4l_id = mfw_gst_get_first_odev();
      }
    }
    /* open the V4l device */
    if ((v4l_info->v4l_id) < 0) {
      GST_ERROR ("Unable to open %s", v4l_info->v4l_dev_name);
      retval = FALSE;
    }



  return retval;

}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_get_crop_cap

DESCRIPTION:        This function get the crop capability value.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_get_crop_cap (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  struct v4l2_cropcap cropcap;
  gboolean retval = TRUE;

  memset (&cropcap, 0, sizeof (cropcap));
  cropcap.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

  if (ioctl (v4l_info->v4l_id, VIDIOC_CROPCAP, &cropcap) < 0) {
    GST_ERROR ("get crop capability failed");
    retval = FALSE;
  }

  return retval;

}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_set_rotation

DESCRIPTION:        This function set rotation and hflip/vflip parameters.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_set_rotation (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  struct v4l2_control ctrl;
  gboolean retval = TRUE;

  /* Set the rotation */
  ctrl.id = V4L2_CID_ROTATE;
  ctrl.value = v4l_info->rotate;
  v4l_info->prevRotate = v4l_info->rotate;
  if (ioctl (v4l_info->v4l_id, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR ("set rotate ctrl failed");
    retval = FALSE;
  }

  /* Set horizontal/vertical flip */
  ctrl.id = V4L2_CID_HFLIP;
  ctrl.value = (v4l_info->hflip == TRUE) ? 1 : 0;
  if (ioctl (v4l_info->v4l_id, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR ("set hflip ctrl failed");
    retval = FALSE;
  }
  ctrl.id = V4L2_CID_VFLIP;
  ctrl.value = (v4l_info->vflip == TRUE) ? 1 : 0;
  if (ioctl (v4l_info->v4l_id, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR ("set vflip ctrl failed");
    retval = FALSE;
  }

  return retval;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_output_setup

DESCRIPTION:        This function set up the display device format

ARGUMENTS PASSED:
        fmt            -   pointer to format for the display device
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE( sucess/failure)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean
mfw_gst_v4l2_output_setup (struct v4l2_format *fmt,
    MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  struct v4l2_requestbuffers buf_req;

  int ret;
  ret = ioctl (v4l_info->v4l_id, VIDIOC_S_FMT, fmt);
  if (ret < 0) {
    GST_ERROR ("set format failed %d", ret);
    return FALSE;
  }

  if (ioctl (v4l_info->v4l_id, VIDIOC_G_FMT, fmt) < 0) {
    GST_ERROR ("get format failed");
    return FALSE;
  }
#if 0                           //test code for sw copy render, also need set MIN_BUFFER_NUM 2
  v4l_info->swbuffer_max = v4l_info->buffers_required - 2;
  v4l_info->buffers_required = 2;

#endif

  while (v4l_info->buffers_required >= MIN_BUFFER_NUM) {

    memset (&buf_req, 0, sizeof (buf_req));
    buf_req.count = v4l_info->buffers_required;
    buf_req.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    buf_req.memory = V4L2_MEMORY_MMAP;


    if (ioctl (v4l_info->v4l_id, VIDIOC_REQBUFS, &buf_req) >= 0) {
      GST_LOG ("%d hwbuffers sucessfully allocated.",
          v4l_info->buffers_required);
      return TRUE;
    }else{
      GST_ERROR("Try to allocate %d hw buffer failed!!", v4l_info->buffers_required);
    }
#ifdef NO_SWBUFFER
    g_print (RED_STR ("Can not allocate enough v4lbuffer\n", 0));

    return FALSE;
#endif
    GST_WARNING ("Fall down to allocate software buffer");
    if (v4l_info->swbuffer_max == 0)
      v4l_info->swbuffer_max = 2;
    v4l_info->swbuffer_max++;
    v4l_info->buffers_required--;
  }

  v4l_info->buffers_required = v4l_info->swbuffer_max = 0;

  return FALSE;
}



/*=============================================================================
FUNCTION:           mfw_gst_v4l2_mx6q_set_fmt

DESCRIPTION:        This function initialise the display device with the specified parameters.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T
        inp_format     -   the display foramt
        width     -   width to be displayed
        height    -   height to be displayed

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_mx6q_set_fmt (MFW_GST_V4LSINK_INFO_T * v4l_info,
    guint in_fmt, guint width, guint height)
{
  struct v4l2_format fmt;
  gboolean retval = TRUE;
  gint i;

  guint in_width = 0;
  guint in_height = 0;
  gint cr_left = 0;
  gint cr_top = 0;
  gint cr_right = 0;
  gint cr_bottom = 0;

  /*No need to set input fmt and req buffer again when change para on-the-fly */
  if (v4l_info->init == FALSE) {
    /* set the input cropping parameters */
    in_width = width;
    in_height = height;

    memset (&fmt, 0, sizeof (fmt));
    fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

    cr_left = v4l_info->cr_left_bypixel;
    cr_top = v4l_info->cr_top_bypixel;
    cr_right = v4l_info->cr_right_bypixel;
    cr_bottom = v4l_info->cr_bottom_bypixel;

    fmt.fmt.pix.width = in_width + cr_left + cr_right;
    fmt.fmt.pix.height = in_height + cr_top + cr_bottom;
    fmt.fmt.pix.pixelformat = in_fmt;


    retval = mfw_gst_v4l2_output_setup (&fmt, v4l_info);
    if (retval == FALSE) {
        GST_ERROR ("Error in mfw_gst_v4lsink_output_setup");
    }
  }
err0:
  return retval;

}



/*=============================================================================
FUNCTION:           mfw_gst_v4l2_set_crop

DESCRIPTION:        This function initialise the crop parameters.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T
        inp_format     -   the display foramt
        disp_width     -   width to be displayed
        disp_height    -   height to be displayed

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_set_crop (MFW_GST_V4LSINK_INFO_T * v4l_info,
    guint display_width, guint display_height)
{
  struct v4l2_control ctrl;
  struct v4l2_crop *crop, *prevCrop;
  struct v4l2_crop newcrop;
  struct v4l2_framebuffer fbuf;
  struct v4l2_format fmt;
  gboolean retval = TRUE;
  struct v4l2_buffer buf;
  gint i;

  guint in_width = 0;
  guint in_height = 0;
  gint cr_left = 0;
  gint cr_top = 0;
  gint cr_right = 0;
  gint cr_bottom = 0;

  crop = &v4l_info->crop;
  prevCrop = &v4l_info->prevCrop;

  mfw_gst_v4l2_get_crop_cap (v4l_info);

  /* set the image rectangle of the display by
     setting the appropriate parameters */
  crop->type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
  crop->c.width = display_width;
  crop->c.height = display_height;
  crop->c.top = v4l_info->axis_top;
  crop->c.left = v4l_info->axis_left;

  if (v4l_info->axis_left < 0) {
      crop->c.left = 0;
      crop->c.width -= -v4l_info->axis_left;
      if (crop->c.width < 0) {
          crop->c.width = 0;
          return retval;
      }
  }
/*  if (v4l_info->axis_left >= v4l_info->fullscreen_width \
      || v4l_info->axis_top >= v4l_info->fullscreen_height) {
    g_print (RED_STR
        ("set v4l display crop sucessfully\n"));
    return retval;
  }*/

  GST_INFO
      ("[V4L Previous Display]: left=%d, top=%d, width=%d, height=%d",
      prevCrop->c.left, prevCrop->c.top, prevCrop->c.width, prevCrop->c.height);

  /* Same with the previous settings, do nothing */
  GST_INFO ("[V4L Current Display]: left=%d, top=%d, width=%d, height=%d",
      crop->c.left, crop->c.top, crop->c.width, crop->c.height);

/*  if ((!memcmp (crop, prevCrop, sizeof (struct v4l2_crop)))
      && (v4l_info->rotate == v4l_info->prevRotate)) {
    GST_INFO ("Same video display crop.\n");
    return TRUE;
  } else*/ {
    memcpy (prevCrop, crop, sizeof (struct v4l2_crop));

    g_print (RED_STR
        ("[V4L Update Display]: left=%d, top=%d, width=%d, height=%d\n",
            crop->c.left, crop->c.top, crop->c.width, crop->c.height));
      mfw_gst_v4l2_streamoff (v4l_info);
  }

  /*
   * before calling S_CROP to setup the destination rectangle, do any
   * framebuffer overlay setup required for it.
   */
  /* set overlay framebuffer pixel format via S_FBUF */
  memset (&fbuf, 0, sizeof (fbuf));
  fbuf.fmt.pixelformat = V4L2_PIX_FMT_RGB565;
  if (v4l_info->galpha != 0)
    fbuf.flags |= V4L2_FBUF_FLAG_GLOBAL_ALPHA;
  if (v4l_info->color_key_en == TRUE)
    fbuf.flags |= V4L2_FBUF_FLAG_SRC_CHROMAKEY;

  if (ioctl(v4l_info->v4l_id, VIDIOC_S_FBUF, &fbuf) < 0) {
    GST_ERROR("VIDIOC_S_FBUF failed\n");
    return FALSE;
  }

  if (ioctl (v4l_info->v4l_id, VIDIOC_S_CROP, crop) < 0) {
    g_print (RED_STR("set crop failed\n"));
	if (ioctl (v4l_info->v4l_id, VIDIOC_G_CROP, crop) < 0)
      GST_ERROR ("get crop failed");

    v4l_info->disp_width = crop->c.width;
    v4l_info->disp_height = crop->c.height;
    v4l_info->axis_left = crop->c.left;
    v4l_info->axis_top = crop->c.top;
    memcpy (prevCrop, crop, sizeof (struct v4l2_crop));
    GST_INFO("Actual crop settings: %d, %d, %d, %d", crop->c.left, crop->c.top,
        crop->c.width, crop->c.height);

    retval = FALSE;
    goto err0;
  } else {
	GST_INFO ("Set crop sucessfully");
    memcpy (prevCrop, crop, sizeof (struct v4l2_crop));
    g_print (RED_STR
        ("[V4L Update Display]: left=%d, top=%d, width=%d, height=%d\n",
         crop->c.left, crop->c.top, crop->c.width, crop->c.height));

    g_print (RED_STR
        ("set v4l display crop sucessfully\n"));
  }

  memset (&fmt, 0, sizeof (fmt));
  fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT_OVERLAY;
  fmt.fmt.win.field = V4L2_FIELD_ANY;
  fmt.fmt.win.chromakey = v4l_info->colorSrc;
  fmt.fmt.win.global_alpha = v4l_info->galpha;
  if (ioctl (v4l_info->v4l_id, VIDIOC_S_FMT, &fmt)) {
    GST_ERROR("VIDIOC_S_FMT failed\n");
    return FALSE;
  }

  newcrop.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

  if (ioctl( v4l_info->v4l_id, VIDIOC_G_CROP, &newcrop) < 0) {
      GST_WARNING("Get crop failed");
      goto err0;
  }
  GST_INFO("Actual crop settings: %d, %d, %d, %d", newcrop.c.left, newcrop.c.top,
          newcrop.c.width, newcrop.c.height);
err0:
  return retval;

}


/*=============================================================================
FUNCTION:           mfw_gst_v4l2_display_init

DESCRIPTION:        This function initialise the display device with
                    the crop parameters.

ARGUMENTS PASSED:
        v4l_info  -   pointer to MFW_GST_V4LSINK_INFO_T
        disp_width     -   width to be displayed
        disp_height    -   height to be displayed

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_display_init (MFW_GST_V4LSINK_INFO_T * v4l_info,
    guint disp_width, guint disp_height)
{

  gboolean retVal = TRUE;

  if ((v4l_info->width == -1) || (v4l_info->height == -1)) {
    GST_WARNING ("Still not get the video information");
    v4l_info->setpara |= PARAM_SET_V4L;
    return FALSE;
  }
  retVal = mfw_gst_v4l2_open (v4l_info);
  if (!retVal)
    return FALSE;

  retVal = mfw_gst_v4l2_set_crop (v4l_info, disp_width, disp_height);
  if (!retVal)
    return FALSE;

  return TRUE;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_input_init

DESCRIPTION:        This function initialise the display device with
                    the specified parameters.

ARGUMENTS PASSED:
        v4l_info  -   pointer to MFW_GST_V4LSINK_INFO_T
        inp_format     -   the display foramt

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_input_init (MFW_GST_V4LSINK_INFO_T * v4l_info, guint inp_format)
{

  gboolean retVal = TRUE;

  guint display_width = 0;
  guint display_height = 0;

  if (v4l_info->chipcode != CC_MX6Q) {
    g_print (RED_STR("only IMX6 chip supported!\n"));
    return FALSE;
  }

  if (v4l_info->init == FALSE)
    v4l_info->querybuf_index = 0;

  v4l_info->frame_dropped = 0;
  v4l_info->drop_frame = FALSE;
  v4l_info->need_check_params = FALSE;

  /* Read the variables passed by the user */

  mfw_gst_v4l2_open (v4l_info);


  if (TRUE == v4l_info->enable_dump)
    dumpfile_open (v4l_info);

  mfw_gst_v4l2_set_rotation (v4l_info);

  mfw_gst_v4l2_mx6q_set_fmt (v4l_info, inp_format,
        v4l_info->width, v4l_info->height);


  return retVal;

}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_streamon

DESCRIPTION:        This function set the v4l2 to streamon

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_streamon (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint type;

  type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
  if (ioctl (v4l_info->v4l_id, VIDIOC_STREAMON, &type) < 0) {
    GST_ERROR ("Could not stream on");
    return FALSE;
  } else {
    GST_INFO ("Set to Stream ON successfully");
  }


  v4l_info->stream_on = TRUE;
  return TRUE;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4l2_streamoff

DESCRIPTION:        This function set the v4l2 to streamoff

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_streamoff (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint type;
  gint err;
  if (v4l_info->stream_on) {
    type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    err = ioctl (v4l_info->v4l_id, VIDIOC_STREAMOFF, &type);
    if (err < 0) {
      g_print ("Set VIDIOC_STREAMOFF failed: %d.\n", err);
      return FALSE;
    } else {
      GST_INFO ("Set to Stream off successfully");
    }

    v4l_info->qbuff_count = 0;
    v4l_info->stream_on = FALSE;
  }
  return TRUE;
}



/*=============================================================================
FUNCTION:           mfw_gst_v4l2_close

DESCRIPTION:        This function close the v4l device

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean
mfw_gst_v4l2_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{

  if (v4l_info->v4l_id != -1) {
    GST_INFO("--> Close v4l2 device");
    close (v4l_info->v4l_id);
    v4l_info->v4l_id = -1;
  }
  return TRUE;
}
